        TITLE TRS-80 Model I Emulator .CAS to SoundBlaster Utility
        ;Copyright (C) 1994 Jeff Vavasour

        NAME CASOUT

        ASSUME CS:PROG,DS:PROG

STACK   SEGMENT STACK
        DB 512 DUP(?)
STACK   ENDS

PROG    SEGMENT

NAME_PTR DW 0                   ;Will contain offset in DTA for filename
LEVEL1  DB 0                    ;Non-zero if using Level I timing constants
SBVOL   DB 0                    ;SoundBlaster volume: 0=loudest, 7=quietest
SB_DETECTED DB 0                ;Changes to 4 if SoundBlaster detected
SBPORT  DW 220H                 ;Base address of SoundBlaster port
ENVIRONMENT DW 0                ;Segment containing environment text

OLDCLKLO DW 0                   ;Offset of original clock routine
OLDCLKHI DW 0                   ;Segment of original clock routine
OLDKEYLO DW 0                   ;Offset of original keyboard routine
OLDKEYHI DW 0                   ;Segment of original keyboard routine

OLDMASK DB 0                    ;Stores original IRQ mask

WAVNAME DB 0,0,64 DUP(0)        ;Storage area for .WAV filename if given

T1_TIME DW 1
T2_TIME DW 1
T3_TIME DW 1
SPACE_TIME DW 1

MSG0    DB 'TRS-80 MODEL I EMULATOR Cassette File Output Utility  '
        DB 'Version 1.2',13,10
        db 'Copyright (C) 1994 Jeff Vavasour',13,10,10,'$'
MSG1    DB 'Usage: CASOUT [/1] [/Vn] [/W] [path\]filename[.CAS]',13,10,10
        DB '       where "filename" is the source virtual cassette file,',13,10
        DB '             "/1" indicates Level I BASIC baud rate,',13,10
        DB '             "/Vn" set volume level "n" (0=quietest, 7=loudest),',13,10
        DB '             "/W" to write to a .WAV file instead of SoundBlaster output.'
        DB 13,10,'$'
MSG2    DB 'Cannot open virtual cassette file by that name.',13,10,'$'
MSG3    DB 'Could not initialise SoundBlaster',13,10,'$'
MSG4    DB 'Press [ESC] to abort. $'
MSG5    DB 13,'End of cassette file.',13,10,'$'
MSG6    DB 'Could not create .WAV file by that name.',13,10,'$'
MSG7    DB 'Error writing to .WAV file!',13,10,'$'
MSG8    DB 'Output file [.WAV]: $'

;34-byte .WAV file header
WAVHEADER DB 'RIFF'
L1LSW   DW 0                    ;File length-8
L1MSW   DW 0
        DB 'WAVEfmt '
        DB 16,0,0,0,1,0,1,0
        DW 2B11H,0,2B11H        ;Sampling frequency and bytes per second
        DB 0,0,1,0,8,0,'data'
L2LSW   DW 0                    ;File length-2CH
L2MSW   DW 0

PARSE:  PUSH DS
        MOV AH,9                ;Display startup message
        PUSH CS
        POP DS
        MOV DX,OFFSET MSG0
        INT 21H
        POP DS
        MOV AX,ES:[2CH]
        MOV CS:ENVIRONMENT,AX
        MOV SI,128              ;Scan command line for filename and parameters
PARSE1: INC SI
        MOV AL,[SI]
        CMP AL,13
        JZ PARSE5
        CMP AL,32
        JZ PARSE1
        CMP BYTE PTR [SI],'/'   ;Look for command line switches
        JZ PARSE2
        CMP CS:NAME_PTR,0
        JNZ PARSE1
        MOV CS:NAME_PTR,SI
        JMP PARSE1
PARSE2: INC SI
        MOV AL,[SI]
        CMP AL,'1'
        JNZ PARSE3
        MOV CS:LEVEL1,-1         ;Level I mode selected, add 1ms space
        JMP PARSE4
PARSE3: AND AL,223
        CMP AL,'V'
        JNZ PARSE4
        INC SI
        MOV AL,[SI]
        SUB AL,'0'
        JB PARSE4
        CMP AL,7
        JA PARSE4
        MOV AH,7                ;Internally SBVOL runs 0 to 7 rather than 7
        SUB AH,AL               ;to 0 (scale is reversed)
        MOV CS:SBVOL,AH
PARSE4: CMP AL,'W'              ;.WAV output option
        JNZ PARSE10
        MOV CS:WAVNAME,60
PARSE10:
        MOV AL,[SI]             ;After a command line switch has been 
        CMP AL,13               ;accounted for, look for next parameter
        JZ PARSE5
        CMP AL,'/'
        JZ PARSE2
        CMP AL,' '
        JZ PARSE1
        INC SI
        JMP PARSE4
PARSE5: MOV SI,CS:NAME_PTR
        MOV BL,0                ;BL will be 0 if .CAS needed, non-zero if not
        CMP SI,0
        JNZ PARSE6
        PUSH CS
        POP DS
        MOV AH,9                ;If no filename specified, display usage
        MOV DX,OFFSET MSG1
        INT 21H
        MOV AX,4C00H            ;and exit
        INT 21H
PARSE6: LODSB
        CMP AL,'/'              ;File name parameter ends with "/", space, or
        JZ PARSE8               ;carriage return
        CMP AL,' '
        JZ PARSE8
        CMP AL,13
        JZ PARSE8
        CMP AL,'.'              ;If a "." detected, an extension was already
        JNZ PARSE7              ;given
        MOV BL,1
        JMP PARSE6
PARSE7: CMP AL,'\'              ;If a "\" detected, the period was part of
        JNZ PARSE6              ;"path" not "filename".
        MOV BL,0
        JMP PARSE6
PARSE8: DEC SI
        CMP BL,0                ;Add .CAS if needed
        JNZ PARSE9
        MOV WORD PTR [SI],432EH ;".CAS"
        MOV WORD PTR [SI+2],5341H
        ADD SI,4
PARSE9: MOV WORD PTR [SI],0     ;Terminating 0 on file name
        MOV AX,3D00H            ;Open file for reading
        MOV DX,CS:NAME_PTR
        INT 21H
        PUSH CS
        POP DS
        JNB FILE_OPENED
        MOV AH,9                ;If error opening file
        MOV DX,OFFSET MSG2
        INT 21H
        MOV AX,4C00H
        INT 21H
FILE_OPENED:
        MOV NAME_PTR,AX         ;Save file's handle
        CMP WAVNAME,0
        JZ NOT_WAV
        CALL OPENWAV
        JMP SB_OK
NOT_WAV:
        CALL SBINIT             ;Initialize SoundBlaster
        TEST SB_DETECTED,-1
        JNZ SB_OK
        MOV AH,9
        MOV DX,OFFSET MSG3
        INT 21H
        MOV AX,4C00H
        INT 21H
SB_OK:  MOV AX,SEG DATA_SEGMENT ;Read up to 65535 bytes into buffer
        MOV BX,NAME_PTR
        MOV DS,AX
        MOV ES,AX
        MOV AH,3FH
        MOV CX,-1
        MOV DX,0
        INT 21H
        MOV CX,AX               ;CX will contain the actual length
        MOV AH,3EH              ;Close file
        INT 21H
        PUSH CS
        POP DS
        CALL SHIFT_BITS         ;Make sure bytes are aligned correctly
        CMP WAVNAME,0
        JNZ WAV_OUTPUT
        PUSH ES
        MOV AX,3508H            ;Store old clock interrupt
        INT 21H
        MOV OLDCLKLO,BX
        MOV OLDCLKHI,ES
        MOV AX,3509H            ;Store old keyboard interrupt
        INT 21H
        MOV OLDKEYLO,BX
        MOV OLDKEYHI,ES
        POP ES
        MOV AH,9
        MOV DX,OFFSET MSG4
        INT 21H
        PUSH CX
        CALL CALIBRATE          ;Work out timing constants
        POP CX
        IN AL,33
        MOV OLDMASK,AL
        OR AL,1
        OUT 33,AL
        MOV AX,2509H            ;Enable new keyboard routine
        MOV DX,OFFSET KEYBOARD
        INT 21H
WAV_OUTPUT:
        JCXZ FINISHED           ;If file null-length, the exit
        MOV SI,0
        CALL PLAY_BYTE          ;Send one byte
        DEC CX
        JZ FINISHED
OUTPUT_HEADER:                  ;Output the leading zeroes
        CMP BYTE PTR ES:[SI],0
        JNZ OUTPUT_MAIN
        CALL PLAY_BYTE
        CMP BYTE PTR ABORT,0
        JNZ DONE
        LOOP OUTPUT_HEADER
        JMP FINISHED
OUTPUT_MAIN:
        CALL PLAY_BYTE          ;Output the sync byte, A5
        DEC CX
        JZ FINISHED
        PUSH CX
        MOV AL,8                ;Pause for 4 bit times (8ms)
PAUSE:  PUSH AX
        CALL SPACE
        POP AX
        DEC AL
        JNZ PAUSE
        POP CX
OUTPUT_LOOP:                    ;Output remainder of file
        CALL PLAY_BYTE
        CMP BYTE PTR ABORT,0
        JNZ DONE
        LOOP OUTPUT_LOOP
FINISHED:                       ;Same as DONE, but prints a message
        MOV AH,9
        MOV DX,OFFSET MSG5
        INT 21H
DONE:   CMP WAVNAME,0
        JNZ WAV_DONE
        MOV AL,OLDMASK          ;Reenable IRQs
        OUT 33,AL
        CALL SBWAIT             ;Shut off SoundBlaster speaker
        MOV AL,0D3H
        OUT DX,AL
        MOV AX,2509H            ;Restore original keyboard handler
        MOV DX,OLDKEYLO
        MOV DS,OLDKEYHI
        INT 21H
        MOV AX,2508H            ;Restore original clock handler
        MOV DX,CS:OLDCLKLO
        MOV DS,CS:OLDCLKHI
        INT 21H
        MOV AX,4C00H            ;Exit to DOS
        INT 21H
WAV_DONE:
        CALL CLOSEWAV
        MOV AX,4C00H
        INT 21H

;Send a single byte of data through the SoundBlaster in Model I cassette
;format

PLAY_BYTE PROC NEAR
        MOV AL,ES:[SI]
        INC SI
        PUSH CX
        MOV CX,8
PLAY_BIT:
        PUSH CX
        PUSH AX
        CALL PULSE              ;Begining of bit: Send clock pulse
        POP AX
        ROL AL,1
        PUSH AX
        JB PLAY_1
        CALL SPACE              ;If bit 0: no pulse for 1ms
        JMP NEXT_BIT
PLAY_1: CALL PULSE              ;If bit 1: send a 1ms pulse
NEXT_BIT:
        POP AX
        POP CX
        LOOP PLAY_BIT
        POP CX
        RET
PLAY_BYTE ENDP

;Send a typical Model I pulse to SoundBlaster

PULSE_DATA DB 255,255,0,0
SPACE_DATA DB 22 DUP(128)

PULSE:
        MOV DX,OFFSET PULSE_DATA
        CMP WAVNAME,0           ;If we're writing a .WAV file, output a
        JNZ WAVPULSE            ;block of data to instead
        MOV AL,1                ;High level output
        CALL CAS_LEVEL
        MOV CX,T1_TIME          ;Duration of 0.16ms
PULSE1: LOOP PULSE1        
        MOV AL,2                ;Low level output
        CALL CAS_LEVEL
        MOV CX,T2_TIME          ;Also duration of 0.16ms
PULSE2: LOOP PULSE2
        MOV AL,0                ;Medium level output
        CALL CAS_LEVEL
        MOV CX,T3_TIME          ;Duration of 0.68ms
PULSE3: LOOP PULSE3
        CMP LEVEL1,0
        JZ PULSE5
        MOV CX,SPACE_TIME       ;If in Level I mode, add an extra 1ms delay
PULSE4: LOOP PULSE4
PULSE5: RET
WAVPULSE:
        PUSH BX                 ;Write the pulse sequence
        MOV AH,40H
        MOV BX,WORD PTR WAVNAME[1]
        MOV CX,11
        CMP LEVEL1,0
        JZ WAVPULSE1
        MOV CL,22               ;Level I baud needs twice the interval length
WAVPULSE1:
        INT 21H
        POP BX
        JB WAVERROR
        RET
WAVERROR:
        MOV AH,9
        MOV DX,OFFSET MSG7
        INT 21H
        JMP DONE

;Generate a space of either 1ms or 2ms depending on baud rate

SPACE   PROC NEAR
        MOV DX,OFFSET SPACE_DATA
        CMP WAVNAME,0           ;If writing a .WAV, write space to file instead
        JNZ WAVPULSE
        MOV CX,SPACE_TIME       ;Duration 1ms
        PUSH CX
SPACE1: LOOP SPACE1
        POP CX
        CMP LEVEL1,0
        JZ SPACE3
SPACE2: LOOP SPACE2             ;Extra 1ms delay if in Level I mode
SPACE3: RET
SPACE   ENDP

;Set DAC level based on value in AL (analogous to bits 1 and 0 of port FFH)

CAS_LEVEL PROC NEAR        
        AND AL,3
        MOV CL,128
        JZ CAS_LEVEL1
        MOV CL,255
        DEC AL
        JZ CAS_LEVEL1
        MOV CL,0
CAS_LEVEL1:
        CALL SBWAIT
        MOV AL,16
        OUT DX,AL
        CALL SBWAIT
        MOV AL,CL
        MOV CL,SBVOL
        SHR AL,CL
        OUT DX,AL
        RET
CAS_LEVEL ENDP

;Null clock routine

INTERRUPT_FLAG DB 0             ;This increments each time CLOCK interrupts
COUNT_LSW DW 0                  ;To contain the least-sig word of loop counter
COUNT_MSW DW 0                  ;...most significant word

CLOCK:  PUSH AX
        INC CS:INTERRUPT_FLAG
        CMP CS:INTERRUPT_FLAG,2
        JNZ CLOCK1
        MOV CS:COUNT_LSW,CX     ;On second interrupt after FLAG reset, store
        MOV CS:COUNT_MSW,DX     ;the counter variables
CLOCK1: MOV AL,32               ;Reset interrupt controller
        OUT 32,AL
        POP AX
        IRET

NULL_CLOCK:                     ;This routine prevents clock interference
        PUSH AX
        MOV AL,32
        OUT 32,AL
        POP AX
        IRET

;Keyboard scan, detects only ESC key

ABORT   DB 0            ;Non-zero if abort requested

KEYBOARD:
        PUSH AX
        IN AL,96
        CMP AL,1
        JNZ KEYBOARD1
        MOV CS:ABORT,-1
KEYBOARD1:
        IN AL,97
        OR AL,128
        OUT 97,AL
        AND AL,127
        OUT 97,AL
        MOV AL,32
        OUT 32,AL
        POP AX
        IRET

;SoundBlaster code:  Identify and initialise

SB_STRING DB 'BLASTER=A'

SBINIT  PROC NEAR
        PUSH ES         ;Check for BLASTER=A in environment
        MOV ES,ENVIRONMENT
        MOV DI,0
        MOV BX,0
SBINIT4:
        CMP BYTE PTR ES:[DI],0
        JZ SBINIT7
        MOV SI,OFFSET SB_STRING
        MOV CX,9
        REPZ CMPSB
        JZ SBINIT6
SBINIT5:
        CMP BYTE PTR ES:[DI-1],0
        JZ SBINIT4
        INC DI
        JMP SBINIT5
SBINIT6:                        ;If it was found, get new port address
        MOV AL,ES:[DI]
        INC DI
        SUB AL,'0'
        CMP AL,10
        JB SBINIT6A
        SUB AL,7
SBINIT6A:
        MOV CL,4
        SHL BX,CL
        MOV AH,0
        ADD BX,AX
        CMP BYTE PTR ES:[DI],'0'
        JNB SBINIT6
        MOV SBPORT,BX
SBINIT7:
        POP ES
        CLI
        MOV DX,SBPORT
        ADD DX,6
        MOV AL,1
        OUT DX,AL
        MOV CX,10
SBINIT1:
        LOOP SBINIT1
        DEC AL
        OUT DX,AL
        STI
        ADD DX,4
        MOV CX,100
SBINIT2:
        IN AL,DX
        CMP AL,0AAH
        JZ SBINIT3
        LOOP SBINIT2
        RET
SBINIT3:                        ;SoundBlaster detected, set mask bit
        MOV SB_DETECTED,4
        CALL SBWAIT
        MOV AL,0D1H             ;Turn on speaker
        OUT DX,AL
        RET
SBINIT  ENDP

;Wait for SoundBlaster command port

SBWAIT  PROC NEAR
        MOV DX,SBPORT
        ADD DX,12
SBWAIT1:
        IN AL,DX
        ROL AL,1
        JB SBWAIT1
        RET
SBWAIT  ENDP

;Intercept the 18.2Hz clock, count number of instructions between interrupts
;to establish delay loop lengths

CALIBRATE PROC NEAR
        MOV AX,2508H
        MOV DX,OFFSET CLOCK
        INT 21H
        MOV CX,0
        MOV DX,0
        MOV INTERRUPT_FLAG,0
CALIBRATE1:                     ;Wait until a clock interrupt occurs
        CMP INTERRUPT_FLAG,0
        JZ CALIBRATE1
CALIBRATE2:
        LOOP CALIBRATE2
        INC DX
        CMP INTERRUPT_FLAG,2    ;Loop until a second interrupt has occurred
        JB CALIBRATE2  
        MOV AX,2508H            ;Disable clock during sound output
        MOV DX,OFFSET NULL_CLOCK
        INT 21H
        MOV DX,COUNT_MSW
        MOV AX,COUNT_LSW
        NOT AX
        CMP AX,1                ;Make sure counter is not 0
        ADC AX,0
        PUSH DX
        PUSH AX
        MOV BX,57               ;1ms=1/55 of a 18.2Hz cycle
        DIV BX
        MOV SPACE_TIME,AX
        MOV T3_TIME,AX
        POP AX
        POP DX
        MOV BX,423              ;0.13ms=1/423 of a 18.2Hz cycle
        DIV BX
        MOV T1_TIME,AX
        MOV T2_TIME,AX
        ADD AX,AX               ;T3=SPACE-T1-T2
        SUB T3_TIME,AX
        RET
CALIBRATE ENDP

;This will make sure that a virtual cassette's bit alignment is correct,
;so that characters are not spread across the lower half of one byte and
;the upper half of another.  This is accomplished by shifting the bits
;until the first non-zero character has a non-zero bit 7.  This character
;should, in fact, be the A5h sync byte.

SHIFT_BITS PROC NEAR
        CMP CX,2                ;There's got to be at least 2 bytes in the
        JNB SHIFT_BITS_1        ;for this to work
        RET
SHIFT_BITS_1:
        PUSH CX
        DEC CX
        MOV DI,1
        MOV AL,0
        REPZ SCASB              ;Look for the first non-zero byte excluding
        POP CX                  ;the first one
        JNZ SHIFT_BITS_2
        RET                     ;If file is just 00's, forget it
SHIFT_BITS_2:
        MOV SI,CX               ;Set pointer to end of file
        DEC SI
        TEST BYTE PTR ES:[DI-1],128
        LAHF
        JZ SHIFT_BITS_3         ;If bit 7 of first non-zero byte is 1, we're
        RET                     ;done
SHIFT_BITS_3:
        SAHF
        RCL BYTE PTR ES:[SI],1  ;Shift every byte left one bit field
        LAHF
        DEC SI
        JNZ SHIFT_BITS_3
        SAHF                    ;Byte at offset 0 is not done by the loop
        RCL BYTE PTR ES:[SI],1
        JMP SHIFT_BITS_2
SHIFT_BITS ENDP

;If a .WAV file is selected for input, get its name

OPENWAV PROC NEAR
        MOV AH,9                ;Prompt for name
        MOV DX,OFFSET MSG8
        INT 21H
        MOV AH,10
        MOV DX,OFFSET WAVNAME
        INT 21H
        MOV AH,2                ;Linefeed after prompt
        MOV DL,10
        INT 21H
        MOV AH,2
        INT 21H
        MOV SI,OFFSET WAVNAME+2 ;Determine if the extension was already
        MOV BL,0                ;given
PARSEWAV:
        LODSB
        CMP AL,'.'
        JNZ PARSEWAV1
        MOV BL,1
PARSEWAV1:
        CMP AL,'\'
        JNZ PARSEWAV2
        MOV BL,0
PARSEWAV2:
        CMP AL,13
        JNZ PARSEWAV
        CMP BL,0
        JNZ ALREADY_EXT         ;Add extension only if not explicitly given
        MOV WORD PTR [SI-1],572EH
        MOV WORD PTR [SI+1],5641H
        ADD SI,4
ALREADY_EXT:
        MOV BYTE PTR [SI-1],0
        MOV AH,3CH              ;Create .WAV file
        MOV CX,0
        MOV DX,OFFSET WAVNAME+2
        INT 21H
        JNB WAV_OK
        MOV DX,OFFSET MSG6
        MOV AH,9
        INT 21H
        MOV AX,4C00H
        INT 21H
WAV_OK: MOV WORD PTR WAVNAME[1],AX
        MOV BX,AX
        MOV AX,4200H            ;Position past .WAV header area
        MOV CX,0                ;That will be written when the file is closed
        MOV DX,2CH
        INT 21H
        MOV CL,SBVOL            ;Adjust volume of pulse sequence data
        MOV SI,OFFSET PULSE_DATA
VOLSET:
        MOV AL,[SI]
        SUB AL,128
        SAR AL,CL
        ADD AL,128
        MOV [SI],AL
        INC SI
        CMP SI,OFFSET SPACE_DATA+22
        JB VOLSET
        RET                     ;Save handle for WAV and return
OPENWAV ENDP

;Close .WAV file and fix length fields in header

CLOSEWAV PROC NEAR
        MOV AX,4202H
        MOV BX,WORD PTR WAVNAME[1]
        MOV CX,0
        MOV DX,0
        INT 21H
        SUB AX,8                ;First field is length-8
        SBB DX,0
        MOV L1LSW,AX
        MOV L1MSW,DX
        SUB AX,24H              ;Second field is length-2CH
        SBB DX,0
        MOV L2LSW,AX
        MOV L2MSW,DX
        MOV AX,4200H            ;Return to beginning of file
        MOV CX,0
        MOV DX,0
        INT 21H
        MOV AH,40H              ;Write header
        MOV CX,2CH
        MOV DX,OFFSET WAVHEADER
        INT 21H
        MOV AH,3EH
        INT 21H
        RET
CLOSEWAV ENDP

PROG    ENDS

DATA_SEGMENT SEGMENT
        DB 0                    ;This is where the .CAS file will be loaded
DATA_SEGMENT ENDS

        END PARSE

