        title TRS-80 Model I Emulator Speed Regulator

        name model1s

        assume cs:prog, ds:prog

stack   segment stack
        db 512 dup(?)
stack   ends

prog    segment

start:

;Step 1: deallocate memory after this program's segment so there's room
;        to load MODEL1.EXE

        mov ax,cs
        mov ds,ax
        mov psp1,es     ;First store PSP address to be passed on
        mov psp2,es
        mov psp3,es
        mov ax,es:[2ch] ;Also store pointer to environment segment
        mov parameter_block,ax

        mov bx,offset free
        mov cl,4
        shr bx,cl
        inc bx
        mov ax,cs
        add bx,ax
        mov ax,es
        sub bx,ax
        mov ah,4ah      ;Free the rest for MODEL1.EXE
        int 21h   
        jnb step2
        jmp error

;Step 2: intercept clock interrupt so that we have an instruction counting
;        source.  And, if desired, disable assignment of COM port IRQ to
;        emulator

step2:
        mov ax,3508h    ;Add self to clock IRQ
        int 21h
        mov int8_lsw,bx
        mov int8_msw,es
        mov ax,2508h
        mov dx,offset clock
        int 21h

        mov ax,3d00h    ;Does "DISABLE.COM" exist?
        mov dx,offset disable_com
        int 21h
        jb step3        ;If not, skip the COM port disable
        mov bx,ax       ;If so, close the file we just opened
        mov ah,3eh
        int 21h
        mov ax,3521h    ;Attach INT 21H and monitor for reassignment
        int 21h         ;of IRQ3 or IRQ4 (INT 0BH or INT 0CH)
        mov int21_lsw,bx
        mov int21_msw,es
        mov ax,2521h
        mov dx,offset int21
        int 21h

;Step 3: intercept video BIOS interrupt (INT 10H) as it's one of the first
;        things called after MODEL1.EXE starts

step3:
        mov ax,3510h
        int 21h
        mov int10_lsw,bx
        mov int10_msw,es
        mov ax,2510h
        mov dx,offset int10
        int 21h
        jmp step4

int10:                  ;We get here via the emulator calling INT 10H
        cmp ax,3
        jz step5
        jmp original_video

;Step 4: load MODEL1.EXE, passing our command line parameters to it.

step4:
        mov ax,cs
        mov es,ax
        mov ax,4b00h
        mov bx,offset parameter_block
        mov dx,offset program_name
        int 21h
        jb run_error
        jmp step10      ;Get here when MODEL1.EXE exits

run_error:              ;Failed to load
        jmp error
        
;Step 5: verify version by looking for strings "Version 3." and
;        "Jeff Vavasour" in the .EXE we just loaded (get here when
;        the emulator calls INT 10H, which is very early and will
;        occur before step 4 returns from its INT 21H).

step5:
        push ax         ;Save registers
        push bx
        push cx
        push dx
        push si
        push di
        push bp
        pushf
        push ds
        push es
        cld

        mov ax,ds       ;Save code segment of MODEL1.EXE (caller)
        mov cs:model1_msw,ax
        mov es,ax

        mov ax,2510h    ;Restore INT 10H to its original vector first
        mov dx,cs:int10_lsw
        mov ds,cs:int10_msw
        int 21h

        mov ax,cs
        mov ds,ax

        mov si,offset name_string
        call find_string
        jb wrong_version
        mov si,offset version_string
        call find_string
        jnb step6

wrong_version:
        mov ax,0
        jmp error

;Step 6: find the string "Cannot load MODEL1.FNT", 13, 10, "$" and replace
;        it with the shorter "Need MODEL1.FNT", 13, 10, "$" so there's
;        room to add our long branch

step6:
        mov si,offset target_string
        call find_string
        jb wrong_version

        mov si,offset new_string
replace_string:
        movsb
        cmp byte ptr [si],0
        jnz replace_string

        ;Write JMP FAR vector into our patched instruction
        mov word ptr es:[di],offset vector_handler
        mov word ptr es:[di+2],cs

        mov dx,di       ;dx is going to contain the near vector in
        dec dx          ;MODEL1.EXE to our JMP FAR

;Step 7: make a copy of MODEL1.EXE's vector table

step7:
        mov ds,cs:model1_msw
        mov ax,cs
        mov es,ax
        mov si,0
        mov di,offset vector_table
        mov cx,vector_size
        rep movsb

;Step 8: replace MODEL1.EXE's vector table with one to our long branch

step8:
        mov ax,cs
        mov ds,ax
        mov es,model1_msw
        mov ax,dx
        mov cx,vector_size/2
        mov di,0
        rep stosw

;Step 9: continue the patched MODEL1.EXE

step9:
        pop es          ;Restore registers after INT 10H call
        pop ds
        popf
        pop bp
        pop di
        pop si
        pop dx
        pop cx
        pop bx
        pop ax
original_video:
        db 0eah         ;JMP FAR
int10_lsw dw 0          ;Original INT 10H vector
int10_msw dw 0

;Step 10: returned from MODEL1.EXE so release clock interrupt and exit

step10:
        call restore_interrupts
        mov ax,4c00h
        int 21h

;Vector handler: whenever a Z-80 instruction would be executed we arrive here

vector_handler:
        push bx
        mov bx,cs:[bx+offset vector_table]
        mov cs:model1_lsw,bx
        pop bx
        push bx
        shr bh,1
        mov bl,dl
        mov bl,cs:timing[bx]
        mov bh,0
        sub cs:count,bx
        sbb cs:count[2],0
        jns no_rollover
wait_here:
        cmp byte ptr cs:clock_tick,0
        jz wait_here
        mov byte ptr cs:clock_tick,0
        mov bx,cs:interval
        mov cs:count,bx
        mov bx,cs:interval[2]
        mov cs:count[2],bx
no_rollover:
        pop bx
        db 0eah         ;JMP FAR
model1_lsw dw 0         ;Will be loaded with original instruction vector
model1_msw dw 0         ;MODEL1.EXE's code segment

;Find a string in the program's segment

find_string:
        mov di,0

find_start:
        mov bx,0

find_compare:
        mov al,[si+bx]
        cmp al,0
        jz found
        cmp al,es:[di+bx]
        jnz not_found
        inc bx
        jmp find_compare

not_found:
        inc di
        jnz find_start
        stc
        ret

found:
        clc
        ret

;Clock interrupt handler

clock:
        push ax
        push bx
        push es
        mov ax,0
        mov es,ax               ;Get vector to model1.exe's IRQ handler
        mov bx,word ptr es:[20h]
        mov ax,word ptr es:[22h]
        cmp ax,cs:model1_msw    ;If INT 8's segment is not model1.exe, skip
        jnz no_change
        mov es,ax               ;Patch original handler to stop hitting
                                ;timer redundantly every tick
        mov byte ptr es:[bx+11],0ebh
        mov bx,es:[bx+3]        ;Within handler is a pointer to the CYCLES
        mov al,es:[bx]          ;variable.  Get pointer and read variable
        and al,127
        cmp al,2                ;User selected 40Hz?
        jnz normal_irq
        mov cs:interval,44325   ;Clock is 40Hz = 44325 Model I CPU cycles
        mov cs:interval[2],0
        mov bx,747ah
        jmp check_acceleration
normal_irq:
        mov cs:interval,31882   ;Clock is 18.2Hz = 97418 Model I CPU cycles
        mov cs:interval[2],1
        mov bx,0ffffh
check_acceleration:
        cmp ax,cs:accelerate    ;Has clock acceleration changed?
        jz no_change
        mov cs:accelerate,ax
        mov al,36h              ;Restore original interval timer (fixes bug)
        out 43h,al
        mov al,bl
        out 40h,al
        mov al,bh
        out 40h,al
no_change:                      ;Set tick flag indicating interval passed
        mov byte ptr cs:clock_tick,1
        pop es
        pop bx
        pop ax
        db 0eah                 ;JMP FAR
int8_lsw dw 0                   ;Original INT 8 vector
int8_msw dw 0

;Restore interrupts

restore_interrupts:
        sti
        mov dx,cs:int21_lsw
        mov ds,cs:int21_msw
        mov ax,ds
        or ax,dx
        jz int21_ok
        mov ax,2521h
        int 21h
int21_ok:
        mov dx,cs:int10_lsw
        mov ds,cs:int10_msw
        mov ax,ds
        or ax,dx
        jz int10_ok
        mov ax,2510h
        int 21h
int10_ok:
        mov dx,cs:int8_lsw
        mov ds,cs:int8_msw
        mov ax,ds
        or ax,dx
        jz int8_ok
        mov ax,2508h
        int 21h
int8_ok:
        cli
        ret

;Interrupt 21H intercept which disables COM port modification

int21:
        push bx
        push bp
        mov bp,sp       ;Was calling segment MODEL1.EXE?
        mov bx,[bp+6]
        pop bp
        cmp bx,cs:model1_msw
        jnz not_relevant
        mov bx,ax       ;Was it a COM port?
        sub bx,250bh
        cmp bx,2
        jnb not_relevant
        pop bx
        iret            ;If so, do nothing.
not_relevant:           ;If not, nothing to see here.  Move along.
        pop bx
        db 0eah
int21_lsw dw 0
int21_msw dw 0

;Error handler: print error message and exit

error:  push ax
        call restore_interrupts
        pop ax
        push ax
        mov dx,offset error_unknown
        mov bx,ax
        add bx,bx
        add bx,offset error_list
        cmp bx,offset end_error_list
        jnb error_message
        cmp word ptr cs:[bx],0
        jz error_message
        mov dx,cs:[bx]
error_message:
        push cs
        pop ds
        mov ah,9
        int 21h
        pop ax
        mov ah,4ch
        int 21h

;Clock variables

clock_tick db 0
interval dw 44325,0
count   dw 0,0
accelerate dw 1234      ;Initial value forces acceleration to be modified

;Error messages

name_string db 'Jeff Vavasour',0
version_string db 'Version 3.',0
target_string db 'Cannot load MODEL1.FNT',10,13,'$',0
new_string db 'Need MODEL1.FNT',10,13,'$',0eah,0

error_unknown db 'Unknown error',13,10,'$'

error0  db 'This program only works with Version 3.x of Jeff Vavasour',27h
        db 's MODEL1.EXE',13,10,'$'
error1  db 'Invalid function',13,10,'$'
error2  db 'MODEL1.EXE not found',13,10,'$'
error7  db 'Invalid block',13,10,'$'
error8  db 'Arena trashed',13,10,'$'
error9  db 'Not enough memory',13,10,'$'
error10 db 'Bad environment',13,10,'$'
error11 db 'Invalid function',13,10,'$'

error_list dw error0,error1,error2,0,0,0,0,error7
        dw error8,error9,error10,error11
end_error_list equ $

;.EXE launch parameters

program_name db 'model1.exe',0

disable_com db 'disable.com',0

parameter_block dw 0    ;Segment of environment string
        dw 80h          ;Program segment prefix's data at offset 80H
psp1    dw 0            ;This .EXE's PSP segment goes here
        dw 5ch          ;Program segment prefix's data at offset 5CH
psp2    dw 0            ;This .EXE's PSP segment goes here
        dw 6ch          ;Program segment prefix's data at offset 6CH
psp3    dw 0            ;This .EXE's PSP segment goes here

;Timing tables

        include timing.inc

vector_table equ $
vector_size equ 1088
free    equ $+vector_size

prog    ends

        end start
